# Bombardier CRJ700 series
# Aircraft systems
##########################

# NOTE: This file contains a loop for running all update functions, so it should be loaded last

## Properties to save on exit
var save_properties =
 [
 "instrumentation/eicas[0]/page",
 "instrumentation/eicas[1]/page",
 "instrumentation/mfd[0]/page",
 "instrumentation/mfd[0]/tcas",
 "instrumentation/mfd[0]/wx",
 "instrumentation/mfd[1]/page",
 "instrumentation/mfd[1]/tcas",
 "instrumentation/mfd[1]/wx",
 "instrumentation/use-metric-altitude",
 "controls/gear/enable-tiller",
 "sim/model/yokes"
 ];
foreach (var property; save_properties)
 {
 aircraft.data.add(property);
 }

## Main systems update loop
var systems =
 {
 loopid: -1,
 init: func
  {
  print("CRJ700 aircraft systems ... initialized");
  systems.loopid += 1;
  settimer(func systems.update(systems.loopid), 0);

  # create crossfeed valves
  var gravity_xflow = aircraft.crossfeed_valve.new(0.5, "controls/fuel/gravity-xflow", 0, 1);
  gravity_xflow.open();
  },
 stop: func
  {
  systems.loopid -= 1;
  },
 reinit: func
  {
  print("CRJ700 aircraft systems ... reinitialized");
  setprop("sim/model/start-idling", 0);
  systems.stop();
  settimer(func systems.update(systems.loopid), 0);
  },
 update: func(id)
  {
  # check if our loop id matches the current loop id
  if (id != systems.loopid) return;
  engine1.update();
  engine2.update();
  apu1.update();
  update_electrical();
  update_timers();
  update_tat();
  left_wiper.update();
  right_wiper.update();
  # stop calling our systems code if the aircraft crashes
  if (!props.globals.getNode("sim/crashed").getBoolValue()) settimer(func systems.update(id), 0);
  }
 };

# call init() 2 seconds after the FDM is ready
setlistener("sim/signals/fdm-initialized", func
 {
 settimer(systems.init, 2);
 }, 0, 0);
# call reinit() if the simulator resets
setlistener("sim/signals/reinit", func(reinit)
 {
 if (reinit.getBoolValue())
  {
  systems.reinit();
  }
 }, 0, 0);

## Startup/shutdown functions
var startid = -1;
var startup = func
 {
 startid += 1;
 var id = startid;
 setprop("controls/electric/battery-switch", 1);
 setprop("controls/pneumatic/bleed-source", 2);
 setprop("controls/APU/off-on", 1);
 setprop("controls/engines/engine[0]/cutoff", 0);
 setprop("controls/engines/engine[1]/cutoff", 0);
 settimer(func
  {
  if (id == startid)
   {
   engine1.start();
   engine2.start();
   setprop("controls/electric/engine[0]/generator", 1);
   setprop("controls/electric/engine[1]/generator", 1);
   settimer(func
    {
    if (id == startid)
     {
     setprop("controls/APU/off-on", 0);
     setprop("controls/electric/battery-switch", 0);
     }
    }, 7);
   }
  }, 11);
 };
var shutdown = func
 {
 setprop("controls/engines/engine[0]/cutoff", 1);
 setprop("controls/engines/engine[1]/cutoff", 1);
 setprop("controls/electric/engine[0]/generator", 0);
 setprop("controls/electric/engine[1]/generator", 0);
 };
setlistener("sim/model/start-idling", func(idle)
 {
 var run = idle.getBoolValue();
 if (run)
  {
  startup();
  }
 else
  {
  shutdown();
  }
 }, 0, 0);

## Instant start for tutorials and whatnot
var instastart = func
 {
 setprop("controls/pneumatic/bleed-source", 3);
 setprop("controls/electric/engine[0]/generator", 1);
 setprop("controls/electric/engine[1]/generator", 1);
 setprop("controls/engines/engine[0]/cutoff", 0);
 engine1.start();
 setprop("engines/engine[0]/rpm", 25);
 setprop("controls/engines/engine[1]/cutoff", 0);
 engine2.start();
 setprop("engines/engine[1]/rpm", 25);
 };

## Prevent the gear from being retracted on the ground
setlistener("controls/gear/gear-down", func(v)
 {
 if (!v.getBoolValue())
  {
  var on_ground = 0;
  foreach (var gear; props.globals.getNode("gear").getChildren("gear"))
   {
   var wow = gear.getNode("wow", 0);
   if (wow != nil and wow.getBoolValue()) on_ground = 1;
   }
  if (on_ground) v.setBoolValue(1);
  }
 }, 0, 0);

## Engines at cutoff by default (not specified in -set.xml because that means they cutoffs will be set to 'true' on a reset)
setprop("controls/engines/engine[0]/cutoff", 1);
setprop("controls/engines/engine[1]/cutoff", 1);

## Autopilot
# Basic roll mode controller
var set_ap_basic_roll = func
 {
 var roll = getprop("instrumentation/attitude-indicator[0]/indicated-roll-deg");
 if (math.abs(roll) > 5)
  {
  setprop("controls/autoflight/basic-roll-mode", 0);
  setprop("controls/autoflight/basic-roll-select", roll);
  }
 else
  {
  var heading = getprop("instrumentation/heading-indicator[0]/indicated-heading-deg");
  setprop("controls/autoflight/basic-roll-mode", 1);
  setprop("controls/autoflight/basic-roll-heading-select", heading);
  }
 };
# Basic pitch mode controller
var set_ap_basic_pitch = func
 {
 var pitch = getprop("instrumentation/attitude-indicator[0]/indicated-pitch-deg");
 setprop("controls/autoflight/pitch-select", int((pitch / 0.5) + 0.5) * 0.5);
 };
setlistener("controls/autoflight/lateral-mode", func(v)
 {
 if (v.getValue() == 0) set_ap_basic_roll();
 }, 0, 0);
setlistener("controls/autoflight/vertical-mode", func(v)
 {
 if (v.getValue() == 0 and getprop("controls/autoflight/lateral-mode") != 2) set_ap_basic_pitch();
 }, 0, 0);
setlistener("controls/autoflight/autopilot/engage", func(v)
 {
 if (v.getBoolValue())
  {
  var lat = getprop("controls/autoflight/lateral-mode");
  var ver = getprop("controls/autoflight/vertical-mode");
  if (lat == 0) set_ap_basic_roll();
  if (ver == 0 and lat != 2) set_ap_basic_pitch();
  }
 }, 0, 0);
setlistener("controls/autoflight/flight-director/engage", func(v)
 {
 if (v.getBoolValue())
  {
  var lat = getprop("controls/autoflight/lateral-mode");
  var ver = getprop("controls/autoflight/vertical-mode");
  if (lat == 0) set_ap_basic_roll();
  if (ver == 0 and lat != 2) set_ap_basic_pitch();
  }
 }, 0, 0);

## Instrumentation
# MFD class
var Mfd =
 {
 new: func(n)
  {
  var m = {};
  m.number = n;
  m.page = props.globals.getNode("instrumentation/mfd[" ~ n ~ "]/page", 1);
  m.tcas = props.globals.getNode("instrumentation/mfd[" ~ n ~ "]/tcas", 1);
  m.wx = props.globals.getNode("instrumentation/mfd[" ~ n ~ "]/wx", 1);
  setlistener(m.page, func(v)
   {
   var page = v.getValue();
   var tcas = props.globals.getNode("instrumentation/radar[" ~ m.number ~ "]/display-controls/tcas");
   tcas.setBoolValue(page == 3 ? m.tcas.getBoolValue() : 0);
   var wx = props.globals.getNode("instrumentation/radar[" ~ m.number ~ "]/display-controls/WX");
   wx.setBoolValue(page == 6 ? m.wx.getBoolValue() : 0);
   }, 1, 0);
  }
 };
var Mfd0 = Mfd.new(0);
var Mfd1 = Mfd.new(1);

# Timers and clock
var chrono_timer = aircraft.timer.new("instrumentation/clock/chronometer-time-sec", 1);
var et_timer = aircraft.timer.new("instrumentation/clock/elapsed-time-sec", 1, 0);
setlistener("gear/gear[1]/wow", func(v)
 {
 if (v.getBoolValue())
  {
  et_timer.stop();
  }
 else
  {
  et_timer.start()
  }
 }, 0, 0);

var normtime = func(x)
 {
 while (x >= 60) x -= 60;
 return x;
 };
var update_timers = func
 {
 var chrono_sec = chrono_timer.node.getValue();
 var chrono_node = props.globals.getNode("instrumentation/clock/chronometer-time-fmt", 1);
 if (chrono_sec >= 3600)
  {
  chrono_node.setValue(sprintf("%02.f:%02.f", int(chrono_sec / 3600), normtime(int(chrono_sec / 60))));
  }
 else
  {
  chrono_node.setValue(sprintf("%02.f:%02.f", normtime(int(chrono_sec / 60)), normtime(chrono_sec)));
  }
 var et_sec = et_timer.node.getValue();
 var et_node = props.globals.getNode("instrumentation/clock/elapsed-time-fmt", 1);
 if (et_sec >= 3600)
  {
  et_node.setValue(sprintf("%02.f:%02.f", int(et_sec / 3600), normtime(int(et_sec / 60))));
  }
 else
  {
  et_node.setValue(sprintf("%02.f:%02.f", normtime(int(et_sec / 60)), normtime(et_sec)));
  }
 };
setlistener("sim/time/real/day", func(v)
 {
 # wait one frame to avoid nil property errors
 settimer(func
  {
  var day = v.getValue();
  var month = getprop("sim/time/real/month");
  var year = getprop("sim/time/real/year");

  var date_node = props.globals.getNode("instrumentation/clock/indicated-date-string", 1);
  date_node.setValue(sprintf("%02.f %02.f", day, month));
  var year_node = props.globals.getNode("instrumentation/clock/indicated-short-year", 1);
  year_node.setValue(substr(year ~ "", 2, 4));
  }, 0);
 }, 1, 0);

# Calculate total air temperature
# formula is
#  T = S + (1.4 - 1)/2 * M^2
var update_tat = func
 {
 var node = props.globals.getNode("environment/total-air-temperature-degc", 1);
 var sat = getprop("environment/temperature-degc");
 var mach = getprop("velocities/mach");
 var tat = sat + 0.2 * mach * mach;#math.pow(mach, 2);
 node.setDoubleValue(tat);
 };
# Spool up the instruments every 5 seconds
var update_spin = func
 {
 setprop("instrumentation/attitude-indicator[0]/spin", 1);
 setprop("instrumentation/attitude-indicator[2]/spin", 1);
 setprop("instrumentation/heading-indicator[0]/spin", 1);
 setprop("instrumentation/heading-indicator[1]/spin", 1);
 settimer(update_spin, 5);
 };
settimer(update_spin, 5);

## Wipers
var Wiper =
 {
 new: func(inP, outP, onP, pwrP)
  {
  var m = { parents: [Wiper] };
  m.active = 0;
  m.loopid = -1;
  m.ctl_node = props.globals.getNode(inP, 1);
  m.out_node = props.globals.getNode(outP, 1);
  m.on_node = props.globals.getNode(onP, 1);
  m.pwr_node = props.globals.getNode(pwrP, 1);
  return m;
  },
 update: func()
  {
  var switch = me.ctl_node.getValue();
  if (me.pwr_node.getValue() >= 15)
   {
   if (switch == 0)
    {
    me.active = 0;
    me.on_node.setBoolValue(0);
    }
   elsif (!me.active)
    {
    me.active = 1;
    var loopid = me.loopid += 1;
    var time = 1 / switch;
    interpolate(me.out_node, 1, time);
    settimer(func if (loopid == me.loopid) interpolate(me.out_node, 0, time), time);
    }
   elsif (me.out_node.getValue() == 0)
    {
    me.active = 0;
    }
   }
  else
   {
   me.on_node.setBoolValue(switch == 0 ? 0 : 1);
   }
  }
 };
var left_wiper = Wiper.new("controls/anti-ice/wiper[0]", "surface-positions/left-wiper-pos-norm", "controls/anti-ice/wiper-power[0]", "systems/electrical/outputs/wiper[0]");
var right_wiper = Wiper.new("controls/anti-ice/wiper[1]", "surface-positions/right-wiper-pos-norm", "controls/anti-ice/wiper-power[1]", "systems/electrical/outputs/wiper[1]");

## Aircraft-specific dialogs
var dialogs =
 {
 autopilot: gui.Dialog.new("sim/gui/dialogs/autopilot/dialog", "Aircraft/CRJ700-family/Systems/autopilot-dlg.xml"),
 radio: gui.Dialog.new("sim/gui/dialogs/radio-stack/dialog", "Aircraft/CRJ700-family/Systems/radio-stack-dlg.xml"),
 lights: gui.Dialog.new("sim/gui/dialogs/lights/dialog", "Aircraft/CRJ700-family/Systems/lights-dlg.xml"),
 failures: gui.Dialog.new("sim/gui/dialogs/failures/dialog", "Aircraft/CRJ700-family/Systems/failures-dlg.xml"),
 tiller: gui.Dialog.new("sim/gui/dialogs/tiller/dialog", "Aircraft/CRJ700-family/Systems/tiller-dlg.xml")
 };
gui.menuBind("autopilot", "CRJ700.dialogs.autopilot.open();");
gui.menuBind("radio", "CRJ700.dialogs.radio.open();");
